<?php
/**
 * Cron Course Registration Script with RPMS Sync
 * Batch processes all ACTIVE students for a given session and level
 * Syncs courses with RPMS (Record and Performance Management System)
 *
 * Usage:
 *   php cron-course-registration.php --session=2024/2025 --level=100 --semester=HARMATTAN
 *   php cron-course-registration.php --session=2024/2025 --level=100 --semester=RAIN
 *   php cron-course-registration.php --session=2024/2025 --level=100 (processes both semesters)
 *
 * Cron Example (run daily at 2 AM):
 *   0 2 * * * /usr/bin/php /path/to/cron-course-registration.php --session=2024/2025 --level=100 --semester=HARMATTAN >> /var/log/course-reg.log 2>&1
 *
 * Process:
 * 1. Fetches all students with verif = 'ACTIVE' from OLD database
 * 2. For each student, fetches curriculum using pcode, session, level
 * 3. Auto-inserts courses with status 'C' (Core) to NEW database
 * 4. Enrolls student in RPMS if not already enrolled
 * 5. Registers courses with RPMS and updates approval status
 * 6. Sets is_pushed = 1 on all inserted records
 */

// Ensure CLI execution
if (php_sapi_name() !== 'cli') {
    die("This script must be run from command line.\n");
}

error_reporting(E_ALL);
ini_set('display_errors', 1);
date_default_timezone_set('Africa/Lagos');

// Get script directory for relative includes
$scriptDir = dirname(__FILE__);

// Include RPMS webserver functions
require_once($scriptDir . '/service/webserver.php');

// Parse command line arguments
$options = getopt('', ['session:', 'level:', 'semester::', 'limit::', 'dry-run', 'skip-rpms', 'help']);

if (isset($options['help']) || empty($options['session']) || empty($options['level'])) {
    echo <<<HELP
Usage: php cron-course-registration.php [OPTIONS]

Required:
  --session=YYYY/YYYY    Academic session (e.g., 2024/2025)
  --level=NNN            Academic level (100, 200, 300, 400, 500, 600)

Optional:
  --semester=NAME        Semester (HARMATTAN or RAIN). If not specified, processes both.
  --limit=N              Limit number of students to process (for testing)
  --dry-run              Show what would be done without making changes
  --skip-rpms            Skip RPMS sync (only insert to local database)
  --help                 Show this help message

Examples:
  php cron-course-registration.php --session=2024/2025 --level=100 --semester=HARMATTAN
  php cron-course-registration.php --session=2024/2025 --level=200 --dry-run
  php cron-course-registration.php --session=2024/2025 --level=100 --limit=10 --skip-rpms

HELP;
    exit(0);
}

$session = trim($options['session']);
$level = intval($options['level']);
$semester = isset($options['semester']) ? strtoupper(trim($options['semester'])) : null;
$limit = isset($options['limit']) ? intval($options['limit']) : 0;
$dryRun = isset($options['dry-run']);
$skipRpms = isset($options['skip-rpms']);

// Validate inputs
if (!preg_match('/^\d{4}\/\d{4}$/', $session)) {
    die("Error: Invalid session format. Use YYYY/YYYY (e.g., 2024/2025)\n");
}

if (!in_array($level, [100, 200, 300, 400, 500, 600])) {
    die("Error: Invalid level. Must be 100, 200, 300, 400, 500, or 600\n");
}

if ($semester !== null && !in_array($semester, ['HARMATTAN', 'RAIN'])) {
    die("Error: Invalid semester. Must be HARMATTAN or RAIN\n");
}

$semesters = $semester ? [$semester] : ['HARMATTAN', 'RAIN'];

// Database credentials
$old_db_config = [
    'host' => '162.214.100.197',
    'user' => 'OBanjo_m3',
    'pass' => 'jsA95d!7',
    'name' => 'OICT_mission'
];

$new_db_config = [
    'host' => '50.6.4.83',
    'user' => 'eportal_main_student',
    'pass' => '}NGP[4r]pXn+g_^5',
    'name' => 'eportal_main_student'
];

// RPMS API credentials
$rpmsUsername = 'eportal';
$rpmsPassword = 'Ntj)K&eH]waX';

// Allowed RPMS response codes (approved)
$allowedRpmsCodes = ['Z00', 'Z13', 'Z16'];

// Logging function
function logMsg($msg, $type = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    echo "[$timestamp] [$type] $msg\n";
}

// Connect to database
function connectDB($config, $name) {
    logMsg("Connecting to $name database ({$config['host']})...");
    $conn = new mysqli($config['host'], $config['user'], $config['pass'], $config['name']);
    if ($conn->connect_error) {
        logMsg("Connection failed: " . $conn->connect_error, 'ERROR');
        return null;
    }
    $conn->set_charset('utf8');
    logMsg("Connected to $name database successfully.", 'SUCCESS');
    return $conn;
}

// Get or refresh RPMS API key
function getRpmsKey($username, $password) {
    logMsg("Getting RPMS API key...");
    $key = getKey($username, $password);
    if ($key === 'Error' || $key === 'Fault' || strlen($key) < 30) {
        logMsg("Failed to get RPMS API key: $key", 'ERROR');
        return null;
    }
    logMsg("RPMS API key obtained successfully.", 'SUCCESS');
    return $key;
}

// Start processing
logMsg("===============================================");
logMsg("=== CRON COURSE REGISTRATION WITH RPMS SYNC ===");
logMsg("===============================================");
logMsg("Session: $session | Level: $level | Semester(s): " . implode(', ', $semesters));
if ($dryRun) {
    logMsg("DRY RUN MODE - No changes will be made", 'WARNING');
}
if ($skipRpms) {
    logMsg("RPMS SYNC DISABLED - Only local database insertion", 'WARNING');
}
if ($limit > 0) {
    logMsg("Processing limited to $limit students", 'WARNING');
}

// Connect to databases
$old_db = connectDB($old_db_config, 'OLD');
if (!$old_db) {
    die("Failed to connect to OLD database. Exiting.\n");
}

$new_db = null;
if (!$dryRun) {
    $new_db = connectDB($new_db_config, 'NEW');
    if (!$new_db) {
        $old_db->close();
        die("Failed to connect to NEW database. Exiting.\n");
    }
}

// Get RPMS API key
$rpmsKey = null;
if (!$dryRun && !$skipRpms) {
    $rpmsKey = getRpmsKey($rpmsUsername, $rpmsPassword);
    if (!$rpmsKey) {
        logMsg("Continuing without RPMS sync...", 'WARNING');
        $skipRpms = true;
    }
}

// Statistics
$stats = [
    'total_students' => 0,
    'processed' => 0,
    'skipped_no_curriculum' => 0,
    'skipped_already_registered' => 0,
    'courses_inserted' => 0,
    'rpms_enrolled' => 0,
    'rpms_courses_approved' => 0,
    'rpms_courses_rejected' => 0,
    'rpms_errors' => 0,
    'errors' => 0
];

// Cache for curriculum data (pcode => semester => courses)
$curriculumCache = [];

// Function to get curriculum
function getCurriculum($old_db, $pcode, $session, $level, $semester, &$cache) {
    $cacheKey = "{$pcode}_{$session}_{$level}_{$semester}";

    if (isset($cache[$cacheKey])) {
        return $cache[$cacheKey];
    }

    $curriculum = [];

    // Try with exact pcode first
    $stmt = $old_db->prepare("SELECT * FROM curriculum WHERE pcode = ? AND session = ? AND level = ? AND semester = ? ORDER BY status ASC, code ASC");
    $stmt->bind_param("ssis", $pcode, $session, $level, $semester);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        $curriculum[] = $row;
    }
    $stmt->close();

    // If no results, try with session variations
    if (empty($curriculum)) {
        $stmt = $old_db->prepare("SELECT * FROM curriculum WHERE pcode = ? AND level = ? AND semester = ? ORDER BY session DESC, status ASC, code ASC LIMIT 50");
        $stmt->bind_param("sis", $pcode, $level, $semester);
        $stmt->execute();
        $result = $stmt->get_result();

        while ($row = $result->fetch_assoc()) {
            $curriculum[] = $row;
        }
        $stmt->close();
    }

    $cache[$cacheKey] = $curriculum;
    return $curriculum;
}

// Fetch all active students
logMsg("Fetching ACTIVE students from OLD database...");

$studentQuery = "SELECT * FROM students WHERE verif = 'ACTIVE' AND level = ?";
if ($limit > 0) {
    $studentQuery .= " LIMIT $limit";
}

$stmt = $old_db->prepare($studentQuery);
$stmt->bind_param("i", $level);
$stmt->execute();
$studentsResult = $stmt->get_result();

$students = [];
while ($row = $studentsResult->fetch_assoc()) {
    $students[] = $row;
}
$stmt->close();

$stats['total_students'] = count($students);
logMsg("Found {$stats['total_students']} ACTIVE students at level $level");

if ($stats['total_students'] === 0) {
    logMsg("No students to process. Exiting.", 'WARNING');
    $old_db->close();
    if ($new_db) $new_db->close();
    exit(0);
}

// Process each student
$startTime = microtime(true);
$batchSize = 50;

foreach ($students as $index => $student) {
    $studentNum = $index + 1;
    $regnum = $student['regnum'];
    // Strip leading zeros for RPMS matric number
    $matric = ltrim($regnum, '0');

    // Progress indicator
    if ($studentNum % $batchSize === 0 || $studentNum === 1) {
        $elapsed = round(microtime(true) - $startTime, 2);
        $rate = $studentNum > 1 ? round($studentNum / $elapsed, 2) : 0;
        logMsg("Processing student $studentNum/{$stats['total_students']} ($rate/sec) - $regnum");
    }

    // Determine pcode
    $pcode = (!empty($student['ocode'])) ? $student['ocode'] : $student['pcode'];

    // Special handling for NPE at levels 100 and 200
    if ($pcode === 'NPE' && in_array($level, [100, 200])) {
        $pcode = 'ECH';
    }

    // Process each semester
    foreach ($semesters as $sem) {
        // Get curriculum
        $curriculum = getCurriculum($old_db, $pcode, $session, $level, $sem, $curriculumCache);

        // Also try with original pcode if different
        if (empty($curriculum) && $pcode !== $student['pcode']) {
            $curriculum = getCurriculum($old_db, $student['pcode'], $session, $level, $sem, $curriculumCache);
        }

        if (empty($curriculum)) {
            $stats['skipped_no_curriculum']++;
            continue;
        }

        // Filter only core courses
        $coreCourses = array_filter($curriculum, fn($c) => $c['status'] === 'C');

        if (empty($coreCourses)) {
            $stats['skipped_no_curriculum']++;
            continue;
        }

        // Build comma-separated course codes for RPMS
        $courseCodes = implode(',', array_map(fn($c) => $c['code'], $coreCourses));

        if ($dryRun) {
            // Dry run - just count
            $stats['processed']++;
            $stats['courses_inserted'] += count($coreCourses);
            continue;
        }

        try {
            // Check if course_reg exists
            $checkStmt = $new_db->prepare("SELECT sn FROM course_reg WHERE regnum = ? AND session = ? AND semester = ? AND level = ?");
            $checkStmt->bind_param("ssis", $regnum, $session, $sem, $level);
            $checkStmt->execute();
            $existingReg = $checkStmt->get_result()->fetch_assoc();
            $checkStmt->close();

            $regdate = date('Y-m-d H:i:s');
            $isNewRegistration = false;

            if ($existingReg) {
                $creg_id = $existingReg['sn'];

                // Check if courses already registered
                $countStmt = $new_db->prepare("SELECT COUNT(*) as cnt FROM course_regs WHERE creg_id = ?");
                $countStmt->bind_param("i", $creg_id);
                $countStmt->execute();
                $countResult = $countStmt->get_result()->fetch_assoc();
                $countStmt->close();

                if ($countResult['cnt'] > 0) {
                    $stats['skipped_already_registered']++;
                    continue;
                }

                // Update existing record
                $updateStmt = $new_db->prepare("UPDATE course_reg SET regdate = ?, is_pushed = 1 WHERE sn = ?");
                $updateStmt->bind_param("si", $regdate, $creg_id);
                $updateStmt->execute();
                $updateStmt->close();
            } else {
                // Insert new course_reg record
                $insertStmt = $new_db->prepare("INSERT INTO course_reg (regnum, session, semester, level, regdate, stat, is_pushed) VALUES (?, ?, ?, ?, ?, 0, 1)");
                $insertStmt->bind_param("sssis", $regnum, $session, $sem, $level, $regdate);
                $insertStmt->execute();
                $creg_id = $new_db->insert_id;
                $insertStmt->close();
                $isNewRegistration = true;
            }

            // Insert core courses
            $totalUnits = 0;
            $insertedCourses = 0;

            foreach ($coreCourses as $course) {
                // Check if course already exists
                $checkCourse = $new_db->prepare("SELECT id FROM course_regs WHERE creg_id = ? AND ccode = ?");
                $checkCourse->bind_param("is", $creg_id, $course['code']);
                $checkCourse->execute();
                $courseExists = $checkCourse->get_result()->fetch_assoc();
                $checkCourse->close();

                if (!$courseExists) {
                    $insertCourse = $new_db->prepare("INSERT INTO course_regs (creg_id, ccode, course_id, score, rpms, is_pushed) VALUES (?, ?, ?, 0, 0, 1)");
                    $insertCourse->bind_param("isi", $creg_id, $course['code'], $course['sn']);
                    $insertCourse->execute();
                    $insertCourse->close();
                    $insertedCourses++;
                }
                $totalUnits += intval($course['unit']);
            }

            // Update total credits
            $updateTC = $new_db->prepare("UPDATE course_reg SET tc = ? WHERE sn = ?");
            $updateTC->bind_param("ii", $totalUnits, $creg_id);
            $updateTC->execute();
            $updateTC->close();

            $stats['processed']++;
            $stats['courses_inserted'] += $insertedCourses;

            // ============================================
            // RPMS SYNC SECTION
            // ============================================
            if (!$skipRpms && $rpmsKey) {
                try {
                    // Step 1: Check if student is enrolled in RPMS
                    $hasEnrollment = HasEnrollment($matric, $session, $sem, $rpmsKey);

                    // Handle invalid auth key - try to refresh
                    if ($hasEnrollment === 'Invalid Auth Key') {
                        logMsg("RPMS key expired, refreshing...", 'WARNING');
                        $rpmsKey = getRpmsKey($rpmsUsername, $rpmsPassword);
                        if ($rpmsKey) {
                            $hasEnrollment = HasEnrollment($matric, $session, $sem, $rpmsKey);
                        } else {
                            throw new Exception("Failed to refresh RPMS key");
                        }
                    }

                    // Step 2: Enroll student if not already enrolled
                    if ($hasEnrollment !== 'Yes') {
                        $enrollResult = enrollStudent($matric, $session, $sem, $pcode, $level, $rpmsKey);

                        if ($enrollResult === 'Sucessful' || $enrollResult === 'Successful') {
                            $stats['rpms_enrolled']++;
                            logMsg("  Enrolled $matric in RPMS for $session $sem", 'SUCCESS');
                        } elseif (strpos($enrollResult, 'already') !== false) {
                            // Already enrolled is fine
                        } else {
                            logMsg("  Failed to enroll $matric in RPMS: $enrollResult", 'WARNING');
                        }
                    }

                    // Step 3: Register courses with RPMS
                    $registerResult = registerStudentCourses2($matric, $session, $sem, $courseCodes, $rpmsKey, true);

                    if (is_array($registerResult) && isset($registerResult['CourseRegistrationResponse'])) {
                        $allApproved = true;
                        $syncedCourses = [];

                        foreach ($registerResult['CourseRegistrationResponse'] as $courseResponse) {
                            $courseCode = $courseResponse['CourseCode'] ?? '';
                            $responseCode = $courseResponse['ResponseCode'] ?? '';
                            $responseDesc = $courseResponse['ResponseDescription'] ?? '';

                            // Check if approved
                            if (in_array($responseCode, $allowedRpmsCodes)) {
                                $rpmsStatus = 1;
                                $stats['rpms_courses_approved']++;
                                $syncedCourses[] = $courseCode;
                            } else {
                                $rpmsStatus = 0;
                                $allApproved = false;
                                $stats['rpms_courses_rejected']++;
                            }

                            // Update course_regs with RPMS status
                            $updateRpms = $new_db->prepare("UPDATE course_regs SET remarks = ?, rpms = ? WHERE ccode = ? AND creg_id = ?");
                            $updateRpms->bind_param("sisi", $responseDesc, $rpmsStatus, $courseCode, $creg_id);
                            $updateRpms->execute();
                            $updateRpms->close();
                        }

                        // Update course_reg sync status
                        $syncStatus = $allApproved ? 'SUCCESSFUL' : 'UNSUCCESSFUL';
                        $syncComment = $allApproved ? 'All courses approved' : 'Some courses rejected';
                        $syncedSerialized = serialize($syncedCourses);

                        $updateSync = $new_db->prepare("UPDATE course_reg SET stat = ?, sync = ?, syncom = ?, syncdate = ?, synced = ? WHERE sn = ?");
                        $statValue = $allApproved ? 1 : 0;
                        $updateSync->bind_param("issssi", $statValue, $syncStatus, $syncComment, $regdate, $syncedSerialized, $creg_id);
                        $updateSync->execute();
                        $updateSync->close();

                    } else {
                        // RPMS returned unexpected result
                        $stats['rpms_errors']++;
                        logMsg("  RPMS returned unexpected result for $matric", 'WARNING');
                    }

                } catch (Exception $e) {
                    $stats['rpms_errors']++;
                    logMsg("  RPMS error for $matric: " . $e->getMessage(), 'ERROR');
                }
            }

        } catch (Exception $e) {
            logMsg("Error processing $regnum: " . $e->getMessage(), 'ERROR');
            $stats['errors']++;
        }
    }
}

// Close connections
$old_db->close();
if ($new_db) $new_db->close();

// Final statistics
$totalTime = round(microtime(true) - $startTime, 2);
$avgRate = $stats['total_students'] > 0 ? round($stats['total_students'] / $totalTime, 2) : 0;

logMsg("===============================================");
logMsg("=== CRON COURSE REGISTRATION COMPLETED ===");
logMsg("===============================================");
logMsg("Duration: {$totalTime}s | Rate: {$avgRate} students/sec");
logMsg("");
logMsg("DATABASE STATISTICS:");
logMsg("  Total Students Found:      {$stats['total_students']}");
logMsg("  Successfully Processed:    {$stats['processed']}");
logMsg("  Skipped (No Curriculum):   {$stats['skipped_no_curriculum']}");
logMsg("  Skipped (Already Reg'd):   {$stats['skipped_already_registered']}");
logMsg("  Total Courses Inserted:    {$stats['courses_inserted']}");
logMsg("  Database Errors:           {$stats['errors']}");
logMsg("");
logMsg("RPMS SYNC STATISTICS:");
logMsg("  Students Enrolled:         {$stats['rpms_enrolled']}");
logMsg("  Courses Approved:          {$stats['rpms_courses_approved']}");
logMsg("  Courses Rejected:          {$stats['rpms_courses_rejected']}");
logMsg("  RPMS Errors:               {$stats['rpms_errors']}");

if ($dryRun) {
    logMsg("");
    logMsg("DRY RUN COMPLETE - No changes were made", 'WARNING');
}

if ($skipRpms) {
    logMsg("");
    logMsg("RPMS SYNC WAS SKIPPED", 'WARNING');
}

exit($stats['errors'] > 0 ? 1 : 0);
